import { useState, useEffect } from 'react';
import Head from 'next/head';

interface Client {
    id: string;
    name: string;
    openaiApiKey: string;
    workflowId: string;
    createdAt: string;
}

export default function AdminDashboard() {
    const [clients, setClients] = useState<Client[]>([]);
    const [showModal, setShowModal] = useState(false);
    const [formData, setFormData] = useState({ name: '', openaiApiKey: '', workflowId: '' });
    const [editingId, setEditingId] = useState<string | null>(null);

    useEffect(() => {
        fetchClients();
    }, []);

    const fetchClients = async () => {
        const res = await fetch('/api/clients');
        if (res.ok) {
            const data = await res.json();
            setClients(data);
        }
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        const method = editingId ? 'PUT' : 'POST';
        const url = editingId ? `/api/clients/${editingId}` : '/api/clients';

        const res = await fetch(url, {
            method,
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(formData),
        });

        if (res.ok) {
            setShowModal(false);
            setFormData({ name: '', openaiApiKey: '', workflowId: '' });
            setEditingId(null);
            fetchClients();
        }
    };

    const handleDelete = async (id: string) => {
        if (confirm('Are you sure you want to delete this client?')) {
            const res = await fetch(`/api/clients/${id}`, { method: 'DELETE' });
            if (res.ok) fetchClients();
        }
    };

    const handleEdit = (client: Client) => {
        setFormData({ name: client.name, openaiApiKey: client.openaiApiKey, workflowId: client.workflowId });
        setEditingId(client.id);
        setShowModal(true);
    };

    const getEmbedCode = (clientId: string) => {
        const code = `<iframe
  src="${window.location.origin}/chat/${clientId}"
  width="400"
  height="600"
  frameborder="0"
></iframe>`;
        navigator.clipboard.writeText(code);
        alert('Embed code copied to clipboard!');
    };

    return (
        <div className="min-h-screen bg-gray-100 p-8">
            <Head>
                <title>Admin Dashboard - ChatKit</title>
            </Head>

            <div className="max-w-6xl mx-auto">
                <div className="flex justify-between items-center mb-8">
                    <h1 className="text-3xl font-bold text-gray-800">Client Management</h1>
                    <button
                        onClick={() => {
                            setFormData({ name: '', openaiApiKey: '', workflowId: '' });
                            setEditingId(null);
                            setShowModal(true);
                        }}
                        className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition"
                    >
                        + Add Client
                    </button>
                </div>

                <div className="bg-white rounded-xl shadow-sm overflow-hidden">
                    <table className="w-full text-left">
                        <thead className="bg-gray-50 border-b">
                            <tr>
                                <th className="p-4 font-semibold text-gray-600">Name</th>
                                <th className="p-4 font-semibold text-gray-600">Workflow ID</th>
                                <th className="p-4 font-semibold text-gray-600">Created At</th>
                                <th className="p-4 font-semibold text-gray-600">Actions</th>
                            </tr>
                        </thead>
                        <tbody className="divide-y">
                            {clients.map((client) => (
                                <tr key={client.id} className="hover:bg-gray-50">
                                    <td className="p-4 font-medium text-gray-900">{client.name}</td>
                                    <td className="p-4 text-gray-600 font-mono text-sm">{client.workflowId}</td>
                                    <td className="p-4 text-gray-500 text-sm">
                                        {new Date(client.createdAt).toLocaleDateString()}
                                    </td>
                                    <td className="p-4 space-x-2">
                                        <button
                                            onClick={() => getEmbedCode(client.id)}
                                            className="text-indigo-600 hover:text-indigo-800 font-medium text-sm"
                                        >
                                            Get Code
                                        </button>
                                        <button
                                            onClick={() => handleEdit(client)}
                                            className="text-blue-600 hover:text-blue-800 font-medium text-sm"
                                        >
                                            Edit
                                        </button>
                                        <button
                                            onClick={() => handleDelete(client.id)}
                                            className="text-red-600 hover:text-red-800 font-medium text-sm"
                                        >
                                            Delete
                                        </button>
                                    </td>
                                </tr>
                            ))}
                            {clients.length === 0 && (
                                <tr>
                                    <td colSpan={4} className="p-8 text-center text-gray-500">
                                        No clients found. Click "Add Client" to get started.
                                    </td>
                                </tr>
                            )}
                        </tbody>
                    </table>
                </div>
            </div>

            {showModal && (
                <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4">
                    <div className="bg-white rounded-xl shadow-xl max-w-md w-full p-6">
                        <h2 className="text-xl font-bold mb-4">
                            {editingId ? 'Edit Client' : 'Add New Client'}
                        </h2>
                        <form onSubmit={handleSubmit} className="space-y-4">
                            <div>
                                <label className="block text-sm font-medium text-gray-700 mb-1">
                                    Client Name
                                </label>
                                <input
                                    type="text"
                                    required
                                    value={formData.name}
                                    onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                                    className="w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 outline-none"
                                    placeholder="e.g. Acme Corp"
                                />
                            </div>
                            <div>
                                <label className="block text-sm font-medium text-gray-700 mb-1">
                                    OpenAI API Key
                                </label>
                                <input
                                    type="password"
                                    required
                                    value={formData.openaiApiKey}
                                    onChange={(e) => setFormData({ ...formData, openaiApiKey: e.target.value })}
                                    className="w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 outline-none"
                                    placeholder="sk-..."
                                />
                            </div>
                            <div>
                                <label className="block text-sm font-medium text-gray-700 mb-1">
                                    Workflow ID
                                </label>
                                <input
                                    type="text"
                                    required
                                    value={formData.workflowId}
                                    onChange={(e) => setFormData({ ...formData, workflowId: e.target.value })}
                                    className="w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 outline-none"
                                    placeholder="ChatKit Workflow ID"
                                />
                            </div>
                            <div className="flex justify-end space-x-3 mt-6">
                                <button
                                    type="button"
                                    onClick={() => setShowModal(false)}
                                    className="px-4 py-2 text-gray-600 hover:bg-gray-100 rounded-lg"
                                >
                                    Cancel
                                </button>
                                <button
                                    type="submit"
                                    className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
                                >
                                    {editingId ? 'Save Changes' : 'Create Client'}
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            )}
        </div>
    );
}
